/*
 * Copyright (c) 2005, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @bug 6255196
 * @summary Verifies the function of method open(java.io.File file).
 * @library /java/awt/regtesthelpers /test/lib
 * @build PassFailJFrame jtreg.SkippedException
 * @run main/manual/othervm OpenTest
 */

import java.awt.Desktop;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import javax.swing.JPanel;

import jtreg.SkippedException;

public class OpenTest extends JPanel {

    static final String INSTRUCTIONS = """
            This test could open the user's home directory and a .txt file.
            After test execution, close the native application windows that
            are used to open the directory and .txt file if they were launched
            by the test.
            If you see any unexpected EXCEPTION messages in the output press Fail.
            Otherwise press Pass.
            """;

    public OpenTest() {
        Desktop desktop = Desktop.getDesktop();

        /*
         * Part 1: open a directory, which should launch the system default
         * file explorer.
         *
         * On Windows platforms, the default file explorer is explorer;
         * on UNIX platforms with Gnome installed and running, the default
         * file explorer is Nautilus.
         */
        File userHome = new File(System.getProperty("user.home"));

        try {
            PassFailJFrame.log("Try to open " + userHome);
            desktop.open(userHome);
            PassFailJFrame.log("Succeed.");
        } catch (IOException e) {
            PassFailJFrame.log("EXCEPTION: " + e.getMessage());
        }

        /*
         * Part 2: open a normal .txt file, which should launch the registered
         * application for .txt files.
         */
        // Create a temp .txt file for test.
        File testFile = null;
        try {
            PassFailJFrame.log("Creating temporary file");
            testFile = File.createTempFile("JDIC-test", ".txt",
                    new File(System.getProperty("java.io.tmpdir")));
            testFile.deleteOnExit();
        } catch (IOException ioe) {
            PassFailJFrame.log("EXCEPTION: " + ioe.getMessage());
            PassFailJFrame.log("Failed to create test file");
        }

        try {
            PassFailJFrame.log("Try to open " + testFile);
            desktop.open(testFile);
            PassFailJFrame.log("Succeed.");
        } catch (IOException e) {
            PassFailJFrame.log("EXCEPTION: " + e.getMessage());
        }
    }

    public static void main(String[] args) throws InterruptedException,
            InvocationTargetException {
        if (!Desktop.isDesktopSupported()) {
            throw new SkippedException("Class java.awt.Desktop is not supported " +
                    "on current platform. Further testing will not be performed");
        }

        PassFailJFrame.builder()
                .title("Mail Test")
                .splitUI(OpenTest::new)
                .instructions(INSTRUCTIONS)
                .rows((int) INSTRUCTIONS.lines().count() + 1)
                .columns(40)
                .logArea()
                .build()
                .awaitAndCheck();
    }
}
